#!/bin/bash

USB_PATH=/mnt/media/${1}
SCRIPT="autoexec.sh"
FILE=$USB_PATH/${SCRIPT}
MANIFEST_FILE=$USB_PATH/manifest.json
SYSTEM_CERT_FILE="/usr/share/platform-keys/platform.pem"
PIDFILE="/var/run/autorun_process.pid"
LOCKFILE="/tmp/__autorun_lock__"
POPUPIDFILE="/tmp/.autorun_msg_id"
RESTORE_DEVICE_FLAG=device-factory-restore

function wait_confirm() {

    # Wait for boot to complete ( SS running )
    while [ ! -e /var/run/jmuconfig.pid ]; do
        sleep 1
    done

    ID=$( DISPLAY=:0 dbus-send --print-reply --session --dest=com.exor.ShowMessage "/" com.exor.ShowMessage.showMessageAsync \
            string:"An autorun script from an external storage has been found.<br>Do you want to execute it?<br>" \
            array:string:"Yes","Cancel" string:"USB Autorun" 2>/dev/null | grep '/ShowMessageInstance/' | sed 's:.*/ShowMessageInstance/\([0-9]\+\)":\1:' )

    [ -z "$ID" ] && exit 0

    echo "$ID:$DEVNAME" > $POPUPIDFILE

    while read lline; do

        echo $lline

        if [ -n "$done" ]; then
            res="$( echo $lline | awk '{ print $2}' )"
            pkill -P $!
            break
        fi

        if ( echo $lline | grep -q "path=/ShowMessageInstance/$ID" ); then
            done=1
            continue
        fi

    done < <( DISPLAY=:0 dbus-monitor --session type='signal',interface=com.exor.ShowMessageInstance,member=done )

    rm $POPUPIDFILE
    pkill -P $$

    [ "$res" != "1" ] && exit 0
}

function close_popup() {
    DISPLAY=:0 dbus-send --print-reply --session --dest=com.exor.ShowMessage "/ShowMessageInstance/$1" com.exor.ShowMessageInstance.close
}

function wait_boot() {
    # wait until the system has boot (no rc scripts running up to 20 seconds)
    for i in `seq 1 20` ; do
        XX="`ps aux`" ; if  ! ( echo $XX | grep "/rc " ) ; then
		break;
	else
		sleep 1
		cat /proc/mounts | grep "$DEVNAME " | grep "/mnt/$1 " || exit
		echo "Waiting system boot complete before starting $SCRIPT script $1 ..." | logger -t "AUTORUN"
	fi
    done
}

function string_rsa_verify() {
    local input="$1"
    local public_key="$2"
    local signature="$3"

    echo "$input" > /tmp/usb_autorun.input

    # hex2bin piped as signature
    if {
        for ((i=0; i<${#signature}; i+=2)); do
            printf "\\x${signature:$i:2}"
        done
    } | openssl dgst -sha256 -verify <(printf "%s" "$public_key") -signature /dev/stdin /tmp/usb_autorun.input; then
        rm -f /tmp/usb_autorun.input
        return 1
    fi

    rm -f /tmp/usb_autorun.input
    return 0
}

function file_rsa_verify() {
    local file_path="$1"
    local public_key="$2"
    local signature="$3"

    # hex2bin piped as signature
    if {
        for ((i=0; i<${#signature}; i+=2)); do
            printf "\\x${signature:$i:2}"
        done
    } | openssl dgst -sha256 -verify <(printf "%s" "$public_key") -signature /dev/stdin $file_path; then
        return 1
    fi

    return 0
}

function verify_manifest() {

    SYS_CERTS_DISABLED=$(sys_params -l factory/options/auth/disableSysCerts)
    SW_COMPS_DISABLED=$(sys_params -l factory/options/auth/disableSigCheck)

    if [[ $SW_COMPS_DISABLED == "true" ]]; then
        echo "No verifications needed: passed successfully"
        return 0
    else
        echo "Need to verify signatures"
    fi

    if [[ ! -f "$MANIFEST_FILE" ]]; then
        echo "Error: Manifest file '$MANIFEST_FILE' does not exist"
        return 1
    fi

    json_content=$(<"$MANIFEST_FILE")

    # get expected JSON signature (first line)
    expected_signature=$(echo "$json_content" | head -n 1 | tr -d '\n' | tr -d '\r')

    json_content_without_signature=$(echo "$json_content" | sed '1d')

    # 'files' array (property of manifest) should contain objects with fields "path" and "signature"
    # since manifest is simple, grep '"path":' and '"signature":'
    file_paths=($(echo "$json_content_without_signature" | grep -oP '"path":\s*"\K[^"]+'))
    file_signatures=($(echo "$json_content_without_signature" | grep -oP '"signature":\s*"\K[^"]+'))

    if [[ ! "${file_paths[@]}" =~ "$SCRIPT" ]]; then
        echo "Sanity check FAILED: $SCRIPT is missing from the file paths"
        return 2
    fi

    if [[ ${#file_paths[@]} -ne ${#file_signatures[@]} ]]; then
        echo "Sanity check FAILED: The number of file paths (${#file_paths[@]}) does not match the number of signatures (${#file_signatures[@]})"
        return 3
    fi

    function signatureVerification() {
        local cert_file="$1"

        # extract public key from the entire certificate
        public_key=$(openssl x509 -in "$cert_file" -pubkey -noout)

        if [[ $? -ne 0 ]]; then
            echo "Error: Public key not obtained correctly from $cert_file"
            return 4
        fi

        # verify signature of entire JSON content
        if string_rsa_verify "$json_content_without_signature" "$public_key" "$expected_signature"; then
            echo "MISMATCH: The entire JSON content does not match the expected signature"
            return 5
        fi

        echo "VERIFIED: The entire JSON content matches the expected signature"

         # flag that tracks verification status
        verification_success=true

        # loop through each file entry in the JSON
        for i in "${!file_paths[@]}"; do
            file_path="${file_paths[$i]}"
            expected_file_signature="${file_signatures[$i]}"

            if file_rsa_verify $USB_PATH/"$file_path" "$public_key" "$expected_file_signature"; then
                echo "MISMATCH: $file_path does not match the signature"
                verification_success=false
            else
                echo "VERIFIED: $file_path matches the signature"
            fi
        done

        if ! $verification_success; then
            return 6
        fi

        return 0
    }

    dbus_result=$(dbus-send --print-reply --system --dest=com.exor.EPAD "/Security" com.exor.EPAD.Security.getSecrets string:'{ "domain": "Public", "secretid": "appCertificate" }')
    USER_CERT_FILE=$(echo "$dbus_result" | grep -o '"publicpath":"[^"]*' | sed 's/"publicpath":"//')

    # check if the publicpath exists as a file
    if [ ! -f "$USER_CERT_FILE" ]; then
        echo "appCertificate file does not exist"
    else
        echo "Using appCertificate certificate $USER_CERT_FILE"

        signatureVerification "$USER_CERT_FILE"
        if [ $? -eq 0 ]; then
            echo "All verifications passed successfully"
            return 0
        fi
    fi

    if [[ $SYS_CERTS_DISABLED != "true" ]]; then
        echo "Using system certificate $SYSTEM_CERT_FILE"

        signatureVerification "$SYSTEM_CERT_FILE"
        if [ $? -eq 0 ]; then
            echo "All verifications passed successfully"
            return 0
        fi
    fi

    echo "Signature verifications failed"
    return 7
}

autorun() {

    # Wait first for rcS to finish to make sure there no pending device configuration
    while ( ps aux | grep -v grep | grep -q /etc/init.d/rcS ); do
        sleep 1
    done

    if [ -e /mnt/${1}/$RESTORE_DEVICE_FLAG -a ! "$(sys_params -l factory/services/usb_device_restore/enabled)" = "false" ]; then
        wait_boot
        dbus-send --print-reply --system --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:880 int32:1000; sleep 1

        rm /mnt/${1}/$RESTORE_DEVICE_FLAG
        sync

        dbus-send --print-reply --system --dest=com.exor.EPAD "/DeviceRestore" com.exor.EPAD.DeviceRestore.restoreDevice
    fi

    if [ -e /mnt/${1}/resetnetworksettings ]; then
        wait_boot
        echo "Network reset requested - fowarding to EPAD" | logger -t "AUTORUN"
        dbus-send --print-reply --system --dest=com.exor.EPAD "/NetworkManager" com.exor.EPAD.NetworkManager.resetConfiguration

        # continuous beep to force user to remove key and reboot
        dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:440 int32:-1
    fi

    if [ -e /mnt/${1}/x5bs.conf ] && [ ! -e /mnt/${1}/x5bs.get ]; then
        echo "Found x5bs.conf, overwrite new list to /etc/x5bs.conf " | logger -t "AUTORUN"
        cp /mnt/${1}/x5bs.conf /etc/x5bs.conf
    fi

    if [ -e /mnt/${1}/x5bs.get ]; then
        if [ -e /etc/x5bs.conf ]; then
            echo "Found x5bs.get, copy /etc/x5bs.conf to /mnt/$1" | logger -t "AUTORUN"
            cp /etc/x5bs.conf /mnt/${1}/x5bs.conf
        else
            echo "/etc/x5bs.conf not exist."
        fi
    fi

    # exit if autoexec is locked by factory setting
    [ "$(/usr/bin/sys_params -r -l factory/services/autorun/mode)" = "locked" ] && exit

    if [ -f ${FILE} ]; then

	# BSP-2890 Detect if it's a USBUpdater
	if [ -e "/mnt/media/${1}/src/updateos.img" ]; then
		# Version is expected somewhere at the beginning of the file.
		# Minimum version allowed is 1.33
		version="$( cat ${FILE} | head -n15 | sed -n 's:^VERSION="\(.*\)\s*":\1:p' )"
		minVers="$( echo -e "1.33\n${version}" | sort -n -t . -k 1 | head -n1 )"

		if [ -z "$version" -o "$minVers" != "1.33" ]; then
			if ! (pidof xsplash &>/dev/null ); then
				DISPLAY=:0 xsplash --no-taptap --no-progress-bar &
				sleep 1
				psplash-write "MSG Unsupported USBUpdater version detected
Aborting..."
				sleep 8
				psplash-write "QUIT"
			fi
			exit 0
		fi
	fi

        echo "usb_autorun manifest verification..." | logger
        verify_manifest
        verify_manifest_exitcode=$?
        if [ $verify_manifest_exitcode -ne 0 ]; then
            echo "Manifest verification exit code $verify_manifest_exitcode";
            echo "usb_autorun manifest verification failed with exit code $verify_manifest_exitcode" | logger
            return $verify_manifest_exitcode
        fi
        echo "usb_autorun manifest verification passed" | logger

        # Show popup before executing
        [ "$(/usr/bin/sys_params -r -l factory/services/autorun/notify)" = "true" ] && wait_confirm

        /bin/bash ${FILE}
    fi
}

if [ "$ACTION" = "remove" ]; then

    if [ -e "$POPUPIDFILE" ]; then
        ID=$( cat $POPUPIDFILE | cut -d':' -f1 )
        DEV=$( cat $POPUPIDFILE | cut -d':' -f2 )

        [ $DEVNAME = "$DEV" ] && close_popup $ID
        rm $POPUPIDFILE
    fi

    exit 0
fi

if [ "${@: -1}" = "start" ]; then
	autorun $@ || exit
	exit 0
fi

# consistecy check : are we executing the script from the deviced signalled by kernel?
cat /proc/mounts | grep "$DEVNAME " | grep "/mnt/media/$1 " || exit

(
	flock -w 1 200 || exit

	# consistency check: avoid running multiple scripts in parallel
	[ -e "$PIDFILE" -a -d "/proc/$( head -n1 "$PIDFILE" )" ] && exit
	[ -e "$PIDFILE" ] && rm "$PIDFILE"

	/usr/sbin/start-stop-daemon -S --background --startas /bin/sh --pidfile $PIDFILE --make-pidfile -- -c "/etc/udev/scripts/usb_autorun.sh $@ start"

) 200>>$LOCKFILE
